<?php
if ( ! defined( 'ABSPATH' ) ) {
	exit; // Exit if accessed directly.
}

class Ultimate_AI_Paypal{
    public $access_token, $payment_data, $UAI_PAYMENTS;
    public function __construct(){
        $this->UAI_PAYMENTS = new Ultimate_AI_Payments_Module();

        if( !$this->UAI_PAYMENTS->gateway_active('paypal') ) return;

        $this->GenerateAccessToken();
    }
    
    public function create_checkout( $data ){
        $this->payment_data = $data;

        if( $this->payment_data['package'] == 'token' ){
            return $this->createOrder();
        }else{
            return $this->createSubscriptionOrder();
        }

    }
    public function createOrder(){
        $args = [
            'method'      => 'POST',
            'data_format' => 'body',
            'headers' => [
                'Content-Type' => "application/json",
                'Authorization'=> 'Bearer ' . $this->access_token,
            ],
            'body' => json_encode([
                "intent" => "CAPTURE",
                "purchase_units" => [[
                    "amount" => [
                        "value" => (int) $this->payment_data['price'],
                        "currency_code" => "USD"
                    ]
                ]],
                "application_context" => [
                    "cancel_url" => $this->UAI_PAYMENTS->getDashboardUrl('pricing'),
                    "return_url" => $this->UAI_PAYMENTS->createSuccessUrl('paypal', $this->payment_data['pricing_id']),
                ] 
            ]),
        ];

        try{
            $response = wp_remote_post( $this->baseURL() . '/v2/checkout/orders', $args );
            $res_body = json_decode( wp_remote_retrieve_body( $response ) );
            
            $error = $this->responseErrorHandler( $response );
            if( $error ){
                return $error;
            }
            return [
                'status' => 200,
                'message' => 'Checkout Session created!',
                'url' => $res_body->links[1]->href,
            ];
        }catch( Exception $e ){
            return [
                'status' => 401,
                'message' => 'Error chckout, ' . $e->getMessage(),
                'url' => null,
            ];
        }
    }

    public function createSubscriptionOrder(){
        $args = [
            'method'      => 'POST',
            'data_format' => 'body',
            'headers' => [
                'Content-Type' => "application/json",
                'Authorization'=> 'Bearer ' . $this->access_token,
            ],
            'body' => json_encode([
                "plan_id" => $this->payment_data['paypal_product_id'],
                "quantity" => '1',
                "application_context" => [
                    "user_action"=> "SUBSCRIBE_NOW",
                    "cancel_url" => $this->UAI_PAYMENTS->getDashboardUrl('pricing'),
                    "return_url" => $this->UAI_PAYMENTS->createSuccessUrl('paypal', $this->payment_data['pricing_id']),
                ] 
            ]),
        ];

        
        try{
            $response = wp_remote_post( $this->baseURL() . '/v1/billing/subscriptions', $args );
            $res_body = json_decode( wp_remote_retrieve_body( $response ) );
            
            $error = $this->responseErrorHandler( $response );
            if( $error ){
                return $error;
            }
            return [
                'status' => 200,
                'message' => 'Checkout Session created!',
                'url' => $res_body->links[0]->href,
            ];
        }catch( Exception $e ){
            return [
                'status' => 401,
                'message' => 'Error chckout, ' . $e->getMessage(),
                'url' => null,
            ];
        }
    }

    public function CreateOrderData( $order_details, $pricing_id ){
        
        $SUBSCRIPTION_MANAGER = new Ultimate_AI_Subscription();

        $package_type = $SUBSCRIPTION_MANAGER->getPackageType( $pricing_id );

        $current_user = wp_get_current_user();

        if( $package_type == 'subscription' ){
            $order = [
                'subscribed'        => true,
                'order_id'          => $order_details->id, //this is the subscription id for paypal
                'subscription_id'   => $order_details->id, //this is the subscription id for paypal
                'user_id'           => get_current_user_id(),
                'name'              => $current_user->display_name,
                'email'             => $current_user->user_email,
                'customer_id'       => $order_details->subscriber->payer_id,
                'starts'            => strtotime($order_details->start_time),
                'expiry'            => strtotime($order_details->billing_info->next_billing_time),
                'payment_method'    => 'paypal',
                'plan_price'        => $order_details->billing_info->last_payment->amount->value,
                'plan_paypal_id'    => $order_details->plan_id,
                'pricing_id'        => $pricing_id,
                'payment_status'    => $order_details->status == 'ACTIVE' ? 'active' : 'inactive',
                'country'           => $order_details->subscriber->shipping_address->address->country_code,
                'address'           => $order_details->subscriber->shipping_address->address,
                'currency'          => $order_details->billing_info->last_payment->amount->currency_code,
            ];
        }elseif( $package_type == 'token' ){
            $order = [
                'subscribed'        => true,
                'user_id'           => get_current_user_id(),
                'order_id'          => $order_details->id,
                'name'              => $current_user->display_name,
                'email'             => $current_user->user_email,
                'address'           => $order_details->payer->address,
                'customer_id'       => $order_details->payer->payer_id,
                'starts'            => time(),
                'expiry'            => 'Unlimited',
                'payment_method'    => 'paypal',
                'plan_price'        => $SUBSCRIPTION_MANAGER->getpackagePrice( $pricing_id ),
                'pricing_id'        => $pricing_id,
                'payment_status'    => $order_details->status == 'COMPLETED' ? 'active' : 'inactive',
                'country'           => $order_details->payer->address->country_code,
                'currency'          => $SUBSCRIPTION_MANAGER->getDefaultCurrency(),
            ];

        }
        return $order;
    }

    public function createProduct( $name ){
        if( !$this->UAI_PAYMENTS->gateway_active('paypal') ) return;

        $args = [
            'method'      => 'POST',
            'headers' => [
                'Content-Type' => "application/json",
                'Authorization'=> 'Bearer ' . $this->access_token,
            ],
            'body' => json_encode([
                "name" => $name,
                "type" => 'DIGITAL'
            ]),
        ];


        try{
            $response = wp_remote_post( $this->baseURL() . '/v1/catalogs/products', $args );
            $res_body = json_decode( wp_remote_retrieve_body( $response ) );
            
            $error = $this->responseErrorHandler( $response );
            if( $error ){
                return $error;
            }
            return [
                'status' => 200,
                'message' => 'Product created!',
                'product_id' => $res_body->id,
            ];
        }catch( Exception $e ){
            return $e->getMessage();
        }
    }

    public function createPricing( $data, $currency ){
        $created_pricing = [];
        if( !$this->UAI_PAYMENTS->gateway_active('paypal') ) return $created_pricing;

 
        foreach( $data['pricing'] as $key => $val ){
            foreach( $val['schemes'] as $k => $scheme ){
              if( !isset( $scheme['paypal_key'] ) || empty( $scheme['paypal_key'] ) ){
                $request_body = [
                    "product_id" => $data['paypal_product_id'],
                    "name" => $data['plan_name'] . '-' . $scheme['type'],
                    "billing_cycles" => [[
                        "tenure_type" => 'REGULAR',
                        "sequence" => 1,
                        "total_cycles" => 0,
                        "frequency" => [
                            "interval_unit" => $scheme['type'] == 'monthly' ? 'MONTH' : 'YEAR',
                            "interval_count"=> 1
                        ],
                        "pricing_scheme" => [
                            "fixed_price" => [
                                "value" => $this->UAI_PAYMENTS->calculate_price($scheme['price'], $scheme['discount'], $scheme['discount_method'] ),
                                "currency_code" => strtoupper($currency)
                            ]
                        ],
                    ]],
                    "payment_preferences" => [
                        "auto_bill_outstanding" => true,
                        "setup_fee_failure_action" => "CONTINUE",
                        "payment_failure_threshold" => 2
                    ]
                ];

                $pricing_obj = $this->createSinglePricing( $request_body );

                if($pricing_obj['status'] == 401){
                    return $pricing_obj;
                }else{
                    $created_pricing[$key][$scheme['type']] = $pricing_obj['pricing_id'];
                }

              }else{
                $created_pricing[$key][$scheme['type']] = $data['pricing'][$key]['schemes'][$k]['paypal_key'];
              }
            }
        }
        return $created_pricing;
    }

    public function createSinglePricing( $request_body ){
        $args = [
            'method'      => 'POST',
            'headers' => [
                'Content-Type' => "application/json",
                'Authorization'=> 'Bearer ' . $this->access_token,
            ],
            'body' => json_encode($request_body),
        ];

        try{
            $response = wp_remote_post( $this->baseURL() . '/v1/billing/plans', $args );
            $res_body = json_decode( wp_remote_retrieve_body( $response ) );

            $error = $this->responseErrorHandler( $response );
            if( $error ){
                return $error;
            }
            return [
                'status' => 200,
                'message' => 'Pricing created!',
                'pricing_id' => $res_body->id,
            ];
        }catch( Exception $e ){
            return $e->getMessage();
        }
    }

    public function responseErrorHandler( $res ){
        $res_body = json_decode( wp_remote_retrieve_body( $res ) );
        if( isset($res_body->message) ){    
            return [
                'status' => 401,
                'message' => $res_body->message 
            ];
        }else{
            return false;
        }
    }

    public function updateProduct( $name, $id ){
        $args = [
            'method'      => 'POST',
            'headers' => [
                'Content-Type' => "application/json",
                'Authorization'=> 'Bearer ' . $this->access_token,
            ],
            'body' => json_encode([[
                "op" => 'replace',
                "path" => '/name',
                "value" => $name
            ]]),
        ];
        try{
            $response = wp_remote_post( $this->baseURL() . '/v1/catalogs/products/' . $id, $args );

            $res_body = json_decode( wp_remote_retrieve_body( $response ) );
            
            $error = $this->responseErrorHandler( $response );
            if( $error ){
                return $error;
            }
            return [
                'status' => 200,
                'message' => 'Product updated!',
                'product_id' => $id,
            ];
        }catch( Exception $e ){
            return $e->getMessage();
        }
    }

    public function capture_order( $order_id ){
        $args = [
            'method'      => 'POST',
            'headers' => [
                'Content-Type' => "application/json",
                'Authorization'=> 'Bearer ' . $this->access_token,
            ]
        ];

        $response = wp_remote_post( $this->baseURL() . '/v2/checkout/orders/' . $order_id . '/capture', $args );

        if ( is_wp_error( $response ) ) {
            $error_message = $response->get_error_message();
            return $error_message;
        } else {
            $res_body = json_decode( wp_remote_retrieve_body( $response ) );
            return $res_body;
        }
    }

    public function order_details( $order_id ){
        $args = [
            'headers' => [
                'Authorization'=> 'Bearer ' . $this->access_token,
            ]
        ];

        $response = wp_remote_get( $this->baseURL() . '/v2/checkout/orders/' . $order_id, $args );

        if ( is_wp_error( $response ) ) {
            $error_message = $response->get_error_message();
            return $error_message;
        } else {
            $res_body = json_decode( wp_remote_retrieve_body( $response ) );
            return $res_body;
        }
    }

    public function subscription_details( $subscription_id ){
        $args = [
            'headers' => [
                'Authorization'=> 'Bearer ' . $this->access_token,
            ]
        ];

        $response = wp_remote_get( $this->baseURL() . '/v1/billing/subscriptions/' . $subscription_id, $args );

        if ( is_wp_error( $response ) ) {
            $error_message = $response->get_error_message();
            return $error_message;
        } else {
            $res_body = json_decode( wp_remote_retrieve_body( $response ) );
            return $res_body;
        }
    }

    public function update_payment_methods(){
        $value = Ultimate_AI_Filter_Controller($_POST['methods']);
        $res = update_option( 'ultimate_ai_payment_method', $value, true );
        wp_send_json( $res, 200 );
    }

    public function get_secret_key(){
        $payment_methods = get_option( 'ultimate_ai_payment_method', [] );
        if(isset($payment_methods['paypal']['gateway'])){
            $gateway = $payment_methods['paypal']['gateway'];
            return $payment_methods['paypal'][$gateway]['api_secret'];

        } else{
            return '';
        }
    }

    public function get_client_id(){
        $payment_methods = get_option( 'ultimate_ai_payment_method', [] );
        if(isset($payment_methods['paypal']['gateway'])){
            $gateway = $payment_methods['paypal']['gateway'];
            return $payment_methods['paypal'][$gateway]['api_key'];

        } else{
            return '';
        }
    }

    public function baseURL(){
        $payment_methods = get_option( 'ultimate_ai_payment_method', [] );
        $gateway = $payment_methods['paypal']['gateway'];

        if($gateway == 'sandbox'){
            return 'https://api-m.sandbox.paypal.com';
        }else{
            return 'https://api-m.paypal.com';
        }
    }

    public function cancelSubscription( $subscription_id ){
        if( !$this->UAI_PAYMENTS->gateway_active('paypal') ) return;

        $args = [
            'method'      => 'POST',
            'headers' => [
                'Authorization'=> 'Bearer ' . $this->access_token,
            ],
            'body' => json_encode([[
                "reason" => 'user cancel'
            ]]),
        ];

        try{
            $response = wp_remote_get( $this->baseURL() . '/v1/billing/subscriptions/' . $subscription_id . '/cancel', $args );

            $res_body = json_decode( wp_remote_retrieve_body( $response ) );
            
            $error = $this->responseErrorHandler( $response );
            if( $error ){
                return $error;
            }
            return [
                'status' => 200,
                'message' => 'Subscription Cancelled'
            ];
        }catch( Exception $e ){
            return $e->getMessage();
        }
    }

    public function GenerateAccessToken(){
        $authString = $this->get_client_id() . ':' .$this->get_secret_key();
        $authString = base64_encode($authString);

        $args = [
            'headers' => [
                'Authorization'=> 'Basic ' . $authString,
            ],
            'body' => "grant_type=client_credentials",
        ];

        $response = wp_remote_post( $this->baseURL() . '/v1/oauth2/token', $args );

        if ( is_wp_error( $response ) ) {
            $error_message = $response->get_error_message();
            return $error_message;
        } else {
            $res_body = json_decode( wp_remote_retrieve_body( $response ) );
            
            if( isset($res_body->error) ){
                return esc_html__( 'Paypal Auth failed', ULTIMATE_AI_SLUG );
            }else{
                return $this->access_token = $res_body->access_token;
            }
        }
    }
}